package de.sillysky.nyssr.impl.example001;


import de.sillysky.nyssr.address.CNodeId;
import de.sillysky.nyssr.address.CSegmentId;
import de.sillysky.nyssr.address.CTargetAddress;
import de.sillysky.nyssr.cli.records.CRecordCliAddHandler;
import de.sillysky.nyssr.cli.records.CRecordCliHandle;
import de.sillysky.nyssr.exception.CException;
import de.sillysky.nyssr.id.common.CWellKnownNID;
import de.sillysky.nyssr.id.common.CWellKnownTID;
import de.sillysky.nyssr.message.CEnvelope;
import de.sillysky.nyssr.namespace.INamespace;
import de.sillysky.nyssr.record.CRecord;
import de.sillysky.nyssr.result.CResultCode;
import de.sillysky.nyssr.service.IService;
import de.sillysky.nyssr.service.IServiceRegistry;
import de.sillysky.nyssr.target.CTarget;
import de.sillysky.nyssr.target.records.CRecordPing;
import de.sillysky.nyssr.target.registry.records.CRecordStartTarget;
import de.sillysky.nyssr.util.cmdline.CSimpleArgParser;
import org.jetbrains.annotations.NotNull;

import java.util.HashMap;
import java.util.Map;

class CServiceExample001 extends CTarget implements IService
{
    private static final String PING = "ping";
    private static final String HELP_PING = //
            "--target [targetAddress]               : ping a target\n" + //
                    "    --nodes [node IDs separated by commas] : ping a target PING.SYSTEM.[nodeId].[local segment id]\n" + //
                    "    -l                                     : list all currently active pings\n" + //
                    "    --delete pingId                        : delete a ping\n" + //
                    "    --delete *                             : delete all pings";

    @NotNull
    private final IDependencies mDependencies;
    private final Map<Long, CPingEntry> mPingEntries = new HashMap<>();

    CServiceExample001(@NotNull final IDependencies aDependencies)
    {
        mDependencies = aDependencies;

        addMessageHandler(CRecordStartTarget.ID,
                          this::asyncStartTarget);
        addMessageHandler(CRecordCliHandle.ID,
                          this::asyncCliHandle);
        addMessageHandler(CRecordPing.ID,
                          this::asyncPing);
    }

    @Override
    public void activate(@NotNull final IServiceRegistry aServiceRegistry) throws Exception
    {
        // We fetch the namespace SYSTEM, which exists in every node.
        final INamespace namespace = mDependencies.getNamespaceRegistry()
                                                  .getNamespace(CWellKnownNID.SYSTEM);
        assert namespace != null;

        // We register the target
        namespace.getTargetRegistry()
                 .registerTarget(this);
    }

    @Override
    public void deactivate(@NotNull final IServiceRegistry aServiceRegistry)
    {
        // We deregister the target
        deregisterTarget();
    }

    private boolean asyncStartTarget(@NotNull final CEnvelope aEnvelope,
                                     @NotNull final CRecord aRecord) throws CException
    {
        if (aEnvelope.isAnswer())
        {
            return false;
        }
        else
        {
            // Register a command line handler
            registerCli();

            aEnvelope.setResultSuccess();
            return true;
        }
    }

    private boolean asyncCliHandle(@NotNull final CEnvelope aEnvelope,
                                   @NotNull final CRecord aRecord) throws CException
    {
        if (aEnvelope.isAnswer())
        {
            return false;
        }
        else
        {
            final String command = CRecordCliHandle.getCommand(aRecord,
                                                               "");
            if (PING.equals(command))
            {
                final String[] arguments = CRecordCliHandle.getArguments(aRecord,
                                                                         null);
                if (arguments.length > 0)
                {
                    final CSimpleArgParser p = new CSimpleArgParser(arguments);

                    final String at = p.getOption("target");
                    if (at != null)
                    {
                        addTargetAddress(at);
                    }
                    final String nodes = p.getOption("nodes");
                    if (nodes != null)
                    {
                        final String[] nodeArr = nodes.split(",");
                        for (final String node : nodeArr)
                        {
                            addNodeId(node);
                        }
                    }
                    final boolean doList = p.hasFlag("l");
                    if (doList)
                    {
                        printList();
                    }
                    final String del = p.getOption("delete");
                    if (del != null)
                    {
                        if ("*".equals(del))
                        {
                            printList();
                            mPingEntries.clear();
                            System.out.println("All ping entries removed");
                        }
                        else
                        {
                            final long id = Long.parseLong(del);
                            final CPingEntry removed = mPingEntries.remove(id);
                            if (removed == null)
                            {
                                System.out.println("ID not found: " + id);
                            }
                            else
                            {
                                System.out.println("Ping entries removed:");
                                System.out.println(createDumpLine(removed));
                            }
                        }
                    }
                }
            }
            aEnvelope.setResultSuccess();
            return true;
        }
    }

    private void addTargetAddress(@NotNull final String aAt) throws CException
    {
        final CTargetAddress targetAddress = CTargetAddress.fromString(aAt);
        if (targetAddress != null)
        {
            final CPingEntry pe = new CPingEntry(targetAddress);
            mPingEntries.put(pe.getId(),
                             pe);
            sendPing(pe);
        }
    }

    private void addNodeId(final String aNode) throws CException
    {
        final CNodeId nodeId = new CNodeId(aNode);
        final CTargetAddress targetAddress = new CTargetAddress(CWellKnownTID.PING,
                                                                CWellKnownNID.SYSTEM,
                                                                nodeId,
                                                                CSegmentId.getLocal());
        final CPingEntry pe = new CPingEntry(targetAddress);
        mPingEntries.put(pe.getId(),
                         pe);
        System.out.println("Adding " + aNode + aNode + ", ID=" + pe.getId());
        sendPing(pe);
    }

    private void printList()
    {
        System.out.printf("%8s %30s %8s %14s %14s%n",
                          "ID",
                          "address",
                          "pings",
                          "pingsPerHour",
                          "msgPerSecond");
        System.out.println("------------------------------------------------------------------------------");

        final StringBuilder sb = new StringBuilder();
        for (final CPingEntry pe : mPingEntries.values())
        {
            sb.append(createDumpLine(pe));
        }
        System.out.println(sb);
    }

    @NotNull
    private String createDumpLine(@NotNull final CPingEntry aEntry)
    {
        return String.format("%8s %30s %8s %,14.0f %,14.0f%n",
                             aEntry.getId(),
                             aEntry.getTargetAddress(),
                             aEntry.getCounter(),
                             aEntry.getPingsPerHour(),
                             aEntry.getPingsPerHour() / 1800);
    }

    private boolean asyncPing(@NotNull final CEnvelope aEnvelope,
                              @NotNull final CRecord aRecord) throws CException
    {
        if (aEnvelope.isAnswer())
        {
            final long id = CRecordPing.getLoopMirror(aRecord,
                                                      0L);
            if (aEnvelope.getResultCode() == CResultCode.SUCCESS)
            {
                final CPingEntry pe = mPingEntries.get(id);
                if (pe != null)
                {
                    pe.incCounter();
                    sendPing(pe);
                }
            }
            else
            {
                mPingEntries.remove(id);
            }

            return true;
        }
        else
        {
            return false;
        }
    }

    private void registerCli() throws CException
    {
        final CEnvelope env = CEnvelope.forLocalNanoService(CWellKnownNID.SYSTEM);
        final CRecord record = CRecordCliAddHandler.create();
        CRecordCliAddHandler.setCommand(record,
                                        new String[]{
                                                PING
                                        });
        CRecordCliAddHandler.setHelp(record,
                                     new String[]{
                                             HELP_PING
                                     });
        sendNotification(env,
                         record);

    }

    private void sendPing(@NotNull final CPingEntry aPingEntry) throws CException
    {
        final CEnvelope env = CEnvelope.forSingleTarget(aPingEntry.getTargetAddress());
        final CRecord record = CRecordPing.create();
        CRecordPing.setLoop(record,
                            aPingEntry.getId());
        sendRequest(env,
                    record);
    }
}
